#include-once
#include <corz_essentials.au3>
global $handle, $zip_loaded, $zipper

; Zip.dll wrapper functions..
;
; These functions are designed to utilize the common capabilities of corz zip.dll
; (as distributed with corz apps installer)
;
; Applications using these functions get drop-in access to all the zipper capabilities.
;
; Cor @ corz.org 2008->tomorrow! ;o)
;


; MakeZip()
;
; Returns an array of two values. The first is the return code from the zipper
; the second is the contents of the @error macro.
;
; Normally, MakeZip() returns a text message describing the state. "ok", if everything
; went okay, or some error message if it didn't.
;
; You can also access the return codes directly, by passing the optional fourth parameter.
; This is handy if you want to intercept them for some reason - you can always use
; ProcessZipCodes() afterwards.
;
; Set the fourth parameter to true, and get back the raw values as an array. This array
; has two values, the first, the actual return code from the zipper, and the second,
; the contents of @error, if there are any.
;
;	$foo = MakeZip($src, $zip, "", 1)
;	$bar = $foo[0]
;	$foo = ProcessZipCodes($foo)
;
; See ProcessZipCodes() below, for a fuller description of the various codes that the
; module creates.
;
func MakeZip($source, $target, $pass="", $raw=0)

	local $result, $ret[2]

	if not $zip_loaded then $zip_loaded = RegisterZipDLL($zipper, $zip_loaded)
	if $zip_loaded < 1 then
		$ret[1] = -2
		return false
	endif


	if $pass then
		$new_zip = ZipCreate($target, $pass) ; $new_zip => handle, expressed as 0x001428**
	else
		$new_zip = ZipCreate($target)
	endif

	; couldn't create zip..
	if Number($new_zip) == 0 then ;	0x00000000
		$result = -8
		SetError(-8)
	else
		; okay..
		if IsDir($source) then
			if StringRight($source, 1) = "\" then $source = StringTrimRight($source, 1)
			$result = ZipAddDir($new_zip, $source, 1)
		else
			$inner = BaseName($source)
			$result = ZipAdd($new_zip, $source, $inner)
		endif
		switch Number($result)
			case 512
				$result = -6
			case 0
				$result = 1
		endswitch
		ZipClose($new_zip)	; = 0

	endif

	$ret[0] = $result
	$ret[1] = @error

	if $raw then return $ret
	return ProcessZipCodes($ret)

endfunc



; UnZip a file, defaults to current location
func UnZip($archive, $destination="", $pass="", $raw=0)

	local $result, $ret[2]

	if not $zip_loaded then $zip_loaded = RegisterZipDLL($zipper, $zip_loaded)
	if $zip_loaded < 1 then return false

	if not $destination then $destination = GetParent($archive)
	$result = ZipUnZip($archive, $destination, $pass)

	$ret[0] = $result
	$ret[1] = @error

	if $raw then return $ret
	return ProcessZipCodes($ret)

endfunc


; RegisterZipDLL()
; get the zipper dll ready for use..
;
; return values:
;	0	=	zipper could not be found
;	1	=	successful loaded corzip plugin [default]
;
func RegisterZipDLL($zipper, $loaded=0)

	if $loaded then return $loaded
	if not FileExists($zipper) then return 0

	$handle = PluginOpen($zipper)
	Sleep(200)
	if PluginClose($handle) then
		$handle = PluginOpen($zipper)
		return 1
	endif
	return -1
endfunc



func ProcessZipCodes($val_arr, $unz=false)

	if not IsArray($val_arr) then return "Not a valid ZipCodes array"

	local $code = $val_arr[0]
	local $err = $val_arr[1]

	local $msg
	if not $zip_loaded then
		$msg = "there was a problem loading the zipper"
	else
		switch $code
			case 0
				if $unz then
					$msg = "ok"
				else
					$msg = "some error"
				endif
			case -6
				$msg = "couldn't add a file to the zip"
			case -8
				$msg = "couldn't create the zip file"
			case else
				; normally 100663296 or something.	[perhaps problem casting in C++] ;2do
				$msg = "ok"
		endswitch
	endif
	return $msg
endfunc



;
; These are from the UDF that came with Stephen Podhajecki's original plugin..
;

;===============================================================================
; Function Name	:	_ZipList2Array
; Description		:	Retrieves and formats the Item Info for all items in the zip
;							into a 2 dimensional array.
; Parameter(s)		:	$szZipFile	The ZipFile to retrieve the item info from.
;
; Requirement(s)	:	Au3Zip.dll Autoit v3.2.2.0 or >
; Return Value(s)	:	2 dimensional array on success for empty string and @error =1
; User CallTip		:
; Author(s)			:	Stephen Podhajecki <gehossafats at netmdc.com/>
; Note(s)			:
;===============================================================================
Func _ZipList2Array($szZipFile)
	Local $zCount, $zipItems[1][8]
	If FileExists($szZipFile) Then
		$zCount = ZipGetCount($szZipFile)
		If $zCount Then
			ReDim $zipItems[$zCount][8]
			For $z_x = 0 To $zCount - 1
				Local $zipTemp = _ZipItemInfo2Array($szZipFile, $z_x)
				If Not (@error) Then
					For $z = 0 To 7
						$zipItems[$z_x][$z] = $zipTemp[$z]
					Next
				EndIf
			Next
		EndIf
		Return $zipItems
	EndIf
	Return SetError(1, 0, "")
EndFunc   ;==>_ZipList2Array


;===============================================================================
; Function Name	:	_ZipItemInfo2Array
; Description		:	Returns and array of zip item info.
; Parameter(s)		:	$szZipFile	The zip file to get the item info from
;							$index		The index of the item to get.
;
; Requirement(s)	:	Au3Zip.dll  Autoit v3.2.2.0
; Return Value(s)	:	An Single dim Array contain the item info, error 1 and empty on fail.
; User CallTip		:
; Author(s)			:	Stephen Podhajecki <gehossafats at netmdc.com/>
; Note(s)			:	This function is called by _ZipList2Array()
;===============================================================================
Func _ZipItemInfo2Array($szZipFile, $index)
	Local $z_v_ret, $strList, $zipTemp, $zipItems[8]
	If FileExists($szZipFile) Then
		If $index >= 0 Then
			If $index <= ZipGetCount ($szZipFile) Then
				$z_v_ret = ZipGetItemInfo ($szZipFile, $index)
				If $z_v_ret <> 0 Then
					Return _ZipGetInfoFromPtr($z_v_ret)
				EndIf
			EndIf
		EndIf
	EndIf
	Return SetError(1, 0, 0)
EndFunc   ;==>_ZipItemInfo2Array


;===============================================================================
; Function Name	:	_ZipGetInfoFromPtr
; Description		:
; Parameter(s)		:	$zPtr	Pointer to the zip entry struct.
;
; Requirement(s)	:	Au3Zip.dll AutoIt v3.2.20
; Return Value(s)	:	Array of zip item info, @error =1 and 0 on fail
; User CallTip		:
; Author(s)			:	Stephen Podhajecki <gehossafats at netmdc.com/>
; Note(s)			:	Called by other fuctions.
;===============================================================================
Func _ZipGetInfoFromPtr($zPtr)
;==============================================================================
;~ typedef struct
;~ { int index;                 // index of this file within the zip
;~   TCHAR name[MAX_PATH];      // filename within the zip
;~   DWORD attr;                // attributes, as in GetFileAttributes.
;~   FILETIME atime,ctime,mtime;// access, create, modify filetimes
;~   long comp_size;            // sizes of item, compressed and uncompressed. These
;~   long unc_size;             // may be -1 if not yet known (e.g. being streamed in)
;~ } ZIPENTRY;
;==============================================================================
	If $zPtr <> 0 Then
		Local $zipItems[8]
		$strList = DllStructCreate("int;char[260];dword;int64;int64;int64;long;long", $zPtr)
		For $z = 1 To 8
			Local $zipTemp = DllStructGetData($strList, $z)
			If $z > 3 And $z < 7 Then
				$zipItems[$z - 1] = _FileTime2SystemTimeZ($zipTemp)
			Else
				$zipItems[$z - 1] = $zipTemp
			EndIf
		Next
		$strList = 0
		Return $zipItems
	EndIf
EndFunc


;===============================================================================
; Function Name:	_FileTime2SystemTimeZ
; Description:		Converts and formats Filetime to Systemtime.
; Parameter(s):			$file_t		64 bit filetime returned from zip archive.
; Requirement(s):
; Return Value(s):	Hopefully a nice formatted string mm/dd/yyyy hr:mim:sec[A/P]m
; User CallTip:
; Author(s):		Stephen Podhajecki <gehossafats at netmdc.com/>
; Note(s):	99% sure this is correct.
;===============================================================================
Func _FileTime2SystemTimeZ($file_t)
	Local $td, $st, $ft, $vret, $syst, $zone, $hr
	$td = "%s/%s/%s %s:%s:%s%s" ;date and time format
	If $file_t = "" Then Return
	$st = DllStructCreate("short;short;short;short;short;short;short;short")
	$ft = DllStructCreate("dword;dword")
	DllStructSetData($ft, 1, _Lo_dWordZ($file_t))
	DllStructSetData($ft, 2, _Hi_dWordZ($file_t))
	$vret = DllCall("kernel32.dll", "int", "FileTimeToSystemTime", "ptr", DllStructGetPtr($ft), "ptr", DllStructGetPtr($st))
	If IsArray($vret) Then
		$zone = "AM"
		$hr = DllStructGetData($st, 5)
		If $hr > 11 Then $zone = "PM"
		If $hr > 12 Then $hr -= 12

		$syst = StringFormat($td, _ZipPZ(DllStructGetData($st, 2)), _
				_ZipPZ(DllStructGetData($st, 4)), _
				DllStructGetData($st, 1), _
				_ZipPZ($hr), _
				_ZipPZ(DllStructGetData($st, 6)), _
				_ZipPZ(DllStructGetData($st, 7)), _
				$zone)

		;ConsoleWrite($syst&@LF)
	EndIf
	$st = 0
	$ft = 0
	Return $syst
EndFunc   ;==>_FileTime2SystemTimeZ
;===============================================================================
; Function Name:	_Hi_dWordZ
; Description:	Get Hi order 32 bits from 64 bit number
; Parameter(s):			$file_t		 filetime
; Requirement(s):
; Return Value(s):	Return Hi order 32 bits from 64 bit number
; User CallTip:
; Author(s):		Stephen Podhajecki <gehossafats at netmdc.com/>
; Note(s):			used to convert filetime to systemtime.
;===============================================================================
Func _Hi_dWordZ($file_t)
	;ConsoleWrite("Hi :"&$file_t/(2^32)&@LF)
	Return $file_t/ (2 ^ 32)
EndFunc   ;==>_Hi_dWordZ
;===============================================================================
; Function Name:	_Lo_dWordZ
; Description:		Gets low order 32 bits of 64 bit number.
; Parameter(s):			$file_t		 Filetime
; Requirement(s):
; Return Value(s):	Lo order 32 bits of 64 bit number.
; User CallTip:
; Author(s):		Stephen Podhajecki <gehossafats at netmdc.com/>
; Note(s):
;===============================================================================
Func _Lo_dWordZ($file_t)
	;ConsoleWrite("Lo:"&abs($file_t - ((2^32) * ($file_t/(2^32))))&@LF)
	Return Abs($file_t - ((2 ^ 32) * ($file_t/ (2 ^ 32))))
EndFunc   ;==>_Lo_dWordZ
;===============================================================================
; Function Name:	_ZipPZ
; Description:		adds a"0" prefix to numbers >=0 and <10
; Parameter(s):			$Value
; Requirement(s):
; Return Value(s):	Modified value.
; User CallTip:
; Author(s):		Stephen Podhajecki <gehossafats at netmdc.com/>
; Note(s):
;===============================================================================
Func _ZipPZ($Value)
	Select
		Case $Value < 0
			Return "00"
		Case $Value < 10
			Return "0" & $Value
	EndSelect
	Return $Value
EndFunc   ;==>_ZipPZ

